// BGG -- I took this from our RTcmix and used it for OSX port of "mix"
// many thanks to John Gibson (and John tells me Chris Penrose)
// for figuring out the mac audio stuff.
// RTcmix on-line: http://music.columbia.edu/RTcmix

/* RTcmix  - Copyright (C) 2000  The RTcmix Development Team
   See ``AUTHORS'' for a list of contributors. See ``LICENSE'' for
   the license to this software and for a DISCLAIMER OF ALL WARRANTIES.
*/

/* For v2.3, by John Gibson. Based on SGI code by Brad Garton and Doug Scott,
   and Linux code by Dave Topper.
   Revised to include OS X support by John Gibson, 11/01.
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <assert.h>
#include "macosx_audio.h"

/* #define MACOSX */

/* #define DEBUG */

/* ------------------------------------------------------- MACOSX support --- */
#ifdef MACOSX
#include <pthread.h>
#include <mach/mach.h>
#include <mach/policy.h>

/* For signalling between main thread and IOProc, which runs as a higher
   priority (IOThread) thread. 
*/
static pthread_cond_t audio_output_condition;
static int buffer_ready = 0;
static float *intermediate_in_buffer, *intermediate_out_buffer;

// BGG the following three were extern but NO LONGER! 
#define MAXBUS 16
BufPtr audioin_buffer[MAXBUS];  /* avoid pulling in all of globals.h */
BufPtr out_buffer[MAXBUS];
int play_audio = 1;

typedef struct {
   int   nfrags;
   int   nframes;
   int   in_nchans;
   int   out_nchans;
   int   indev_nchans;
   int   outdev_nchans;
} DevInfo;
static DevInfo dev_info;

/* ---------------------------------------------------------- audioIOProc --- */
#define IN_GAIN_FACTOR  32768.0
#define OUT_GAIN_FACTOR (1.0 / 32768.0)

static OSStatus
audioIOProc(
      AudioDeviceID           inDevice,
      const AudioTimeStamp    *inNow,
      const AudioBufferList   *inInputData,
      const AudioTimeStamp    *inInputTime,
      AudioBufferList         *outOutputData,
      const AudioTimeStamp    *inOutputTime,
      void                    *devinfo)
{
   int      nsamps;
   float    *in, *out, *tmp;
   DevInfo  *info = (DevInfo *) devinfo;
   static int cur_fragment = 0;
#ifdef DEBUG
   static Float64 lasttime=-1, thistime, timediff;
#endif

   if (!play_audio || !buffer_ready)
      return kAudioHardwareNoError;

#ifdef DEBUG
   thistime = inOutputTime->mSampleTime;
   if (lasttime==-1)
      timediff=0;
   else {
      timediff = thistime - lasttime;
      if (timediff!=info->nframes)
         printf("******timediff != buffer size\n");
   }
   lasttime = thistime;
   printf("ioproc: now=%g, thistime=%g, timediff=%g\n",
               inNow->mSampleTime, thistime, timediff);
   printf("        inchans=%ld, outchans=%ld, "
                              "inbytes=%ld, outbytes=%ld, curfrag=%d\n",
               inInputData->mBuffers[0].mNumberChannels,
               outOutputData->mBuffers[0].mNumberChannels,
               inInputData->mBuffers[0].mDataByteSize,
               outOutputData->mBuffers[0].mDataByteSize,
               cur_fragment);
   fflush(stdout);
#endif

   nsamps = info->nframes * info->outdev_nchans;
   tmp = intermediate_out_buffer + (cur_fragment * nsamps);
//FIXME: What if device has multiple, non-interleaved buffers?
   out = outOutputData->mBuffers[0].mData;
   while (nsamps--)
      *out++ = *tmp++;

   if (info->in_nchans > 0) {
      nsamps = info->nframes * info->indev_nchans;
      tmp = intermediate_in_buffer + (cur_fragment * nsamps);
//FIXME: What if device has multiple, non-interleaved buffers?
      in = inInputData->mBuffers[0].mData;
      while (nsamps--)
         *tmp++ = *in++;
   }

   cur_fragment++;
   if (cur_fragment == info->nfrags) {
      /* unblock main thread waiting on this condition variable */
      pthread_cond_signal(&audio_output_condition);
      cur_fragment = 0;
   }

   return kAudioHardwareNoError;
}


/* ---------------------------------------------------- macosx_audio_read --- */
void
macosx_audio_read()
{
   float *in;
   int   i, j, nsamps;
   static int cur_fragment = 0;

   if (dev_info.in_nchans == 0)
      return;

#ifdef DEBUG
   printf(" read: curfrag=%d\n", cur_fragment);
   fflush(stdout);
#endif

   nsamps = dev_info.nframes * dev_info.indev_nchans;
   in = intermediate_in_buffer + (cur_fragment * nsamps);

   /* Copy from buffer fragment, with appropriate channel mapping. */
   if (dev_info.indev_nchans == dev_info.in_nchans) {
      for (i = 0; i < dev_info.nframes; i++)
         for (j = 0; j < dev_info.in_nchans; j++)
            audioin_buffer[j][i] = *in++ * IN_GAIN_FACTOR;
   }
   else if (dev_info.indev_nchans > dev_info.in_nchans) {
      for (i = 0; i < dev_info.nframes; i++) {
         for (j = 0; j < dev_info.in_nchans; j++)
            audioin_buffer[j][i] = *in++ * IN_GAIN_FACTOR;
         for ( ; j < dev_info.indev_nchans; j++)   /* skip remaining chans */
            in++;
      }
   }
   else if (dev_info.indev_nchans < dev_info.in_nchans) {
      for (i = 0; i < dev_info.nframes; i++) {
         for (j = 0; j < dev_info.in_nchans; j++)
            audioin_buffer[j][i] = *in++ * IN_GAIN_FACTOR;
         for ( ; j < dev_info.in_nchans; j++)      /* zero remaining chans */
            audioin_buffer[j][i] = 0.0;
      }
   }

   cur_fragment++;
   if (cur_fragment == dev_info.nfrags)
      cur_fragment = 0;
}


/* --------------------------------------------------- macosx_audio_write --- */
void
macosx_audio_write()
{
   float *out;
   int   i, j, nsamps;
   static int cur_fragment = 0;
   static pthread_mutex_t audio_output_mutex = PTHREAD_MUTEX_INITIALIZER;

#ifdef DEBUG
   printf(" write: curfrag=%d\n", cur_fragment);
   fflush(stdout);
#endif

   nsamps = dev_info.nframes * dev_info.outdev_nchans;
   out = intermediate_out_buffer + (cur_fragment * nsamps);

   /* Copy into buffer fragment, with appropriate channel mapping. */
   if (dev_info.out_nchans == dev_info.outdev_nchans) {
      for (i = 0; i < dev_info.nframes; i++)
         for (j = 0; j < dev_info.out_nchans; j++)
            *out++ = out_buffer[j][i] * OUT_GAIN_FACTOR;
   }
   else if (dev_info.out_nchans == 1) {
      for (i = 0; i < dev_info.nframes; i++) {
         float samp = out_buffer[0][i] * OUT_GAIN_FACTOR;
         for (j = 0; j < dev_info.outdev_nchans; j++)
            *out++ = samp;
      }
   }
   else {
      for (i = 0; i < dev_info.nframes; i++) {
         for (j = 0; j < dev_info.out_nchans; j++)
            *out++ = out_buffer[j][i] * OUT_GAIN_FACTOR;
         for ( ; j < dev_info.outdev_nchans; j++) /* zero remaining dev chans */
            *out++ = 0.0;
      }
   }

   buffer_ready = 1;
   cur_fragment++;
   if (cur_fragment == dev_info.nfrags) {
      /* init wait condition */
      pthread_cond_init(&audio_output_condition, NULL);
      /* acquire lock */
      pthread_mutex_lock(&audio_output_mutex);
      /* wait until IOProc signals us to continue */
      pthread_cond_wait(&audio_output_condition, &audio_output_mutex);
      /* relinquish lock */
      pthread_mutex_unlock(&audio_output_mutex);
      cur_fragment = 0;
   }
}


/* ------------------------------------------ macosx_cmixplay_audio_write --- */
void
macosx_cmixplay_audio_write(short *buf)
{
   float *out;
   short *in;
   int   i, j, nsamps;
   static int cur_fragment = 0;
   static pthread_mutex_t audio_output_mutex = PTHREAD_MUTEX_INITIALIZER;

#ifdef DEBUG
   printf(" write: curfrag=%d\n", cur_fragment);
   fflush(stdout);
#endif

   nsamps = dev_info.nframes * dev_info.outdev_nchans;
   out = intermediate_out_buffer + (cur_fragment * nsamps);
   in = buf;

   /* Copy into buffer fragment, with appropriate channel mapping. */
   if (dev_info.outdev_nchans == dev_info.out_nchans) {
      for (i = 0; i < nsamps; i++)
         *out++ = *in++ * OUT_GAIN_FACTOR;
   }
   else if (dev_info.out_nchans == 1) {
      for (i = 0; i < dev_info.nframes; i++) {
         float samp = *in++ * OUT_GAIN_FACTOR;
         for (j = 0; j < dev_info.outdev_nchans; j++)
            *out++ = samp;
      }
   }
#if 0
   else {
      for (i = 0; i < dev_info.nframes; i++) {
         for (j = 0; j < dev_info.out_nchans; j++)
            *out++ = out_buffer[j][i] * OUT_GAIN_FACTOR;
         for ( ; j < dev_info.outdev_nchans; j++) /* zero remaining dev chans */
            *out++ = 0.0;
      }
   }
#endif

   buffer_ready = 1;
   cur_fragment++;
   if (cur_fragment == dev_info.nfrags) {
      /* init wait condition */
      pthread_cond_init(&audio_output_condition, NULL);
      /* acquire lock */
      pthread_mutex_lock(&audio_output_mutex);
      /* wait until IOProc signals us to continue */
      pthread_cond_wait(&audio_output_condition, &audio_output_mutex);
      /* relinquish lock */
      pthread_mutex_unlock(&audio_output_mutex);
      cur_fragment = 0;
   }
}


/* ---------------------------------------------------- open_macosx_ports --- */
#define ISINPUT 1
#define ISOUTPUT 0

int
open_macosx_ports(
      int            in_nchans,   /* 0 for no input */
      int            out_nchans,  /* 0 for no output */
      AudioDeviceID  *iport,      /* input device ID */
      AudioDeviceID  *oport,      /* output device ID */
      int            verbose,     /* print buffer sizes, etc. */
      float          srate,
      int            nfrags,      /* number of buffer fragments */
      int            *nframes)    /* number of frames per buffer to request; */
                                  /* on return, contains actual frames */
{
   int         indev_nchans, outdev_nchans, nsamps;
   UInt32      count, in_buffer_size, out_buffer_size, desired_buffer_size;
   float       host_clock_rate;
   uint32_t    period;
   thread_time_constraint_policy_data_t thread_policy;
   char        outdev_name[256];
   OSStatus    err = kAudioHardwareNoError;
   Mac_Boolean     writeable, running, alive;
   AudioDeviceID  indevice, outdevice;
   AudioStreamBasicDescription devFormat;

   if (!play_audio)
      return 0;

   indevice = outdevice = kAudioDeviceUnknown;
   if (iport)
      *iport = kAudioDeviceUnknown;
   if (oport)
      *oport = kAudioDeviceUnknown;

   /* Get default audio output device. */
   count = sizeof(outdevice);
   err = AudioHardwareGetProperty(kAudioHardwarePropertyDefaultOutputDevice,
               &count, (void *) &outdevice);
   if (err != kAudioHardwareNoError || outdevice == kAudioDeviceUnknown) {
      fprintf(stderr, "Can't find default audio output device (error %ld).\n",
                                                                         err);
      return -1;
   }

#ifdef NOMORE
   /* Get audio device name. */
   count = 256;
   err = AudioDeviceGetProperty(outdevice, 0, ISOUTPUT,
                     kAudioDevicePropertyDeviceName, &count, outdev_name);
   assert(err == kAudioHardwareNoError);
   printf("Audio output device: \"%s\"\n",  outdev_name);
#endif

#ifdef NOT_HERE
// FIXME: Do this from a notification callback?
   /* Make sure device is alive (connected). */
   count = sizeof(UInt32);
   err = AudioDeviceGetProperty(outdevice, 0, ISOUTPUT,
                     kAudioDevicePropertyDeviceIsAlive, &count, &alive);
   assert(err == kAudioHardwareNoError);
   if (!alive) {
      fprintf(stderr, "Audio output device not connected.\n");
      return -1;
   }

   /* Make sure device is running.  (What does this mean, exactly?) */
   count = sizeof(UInt32);
   err = AudioDeviceGetProperty(outdevice, 0, ISOUTPUT,
                     kAudioDevicePropertyDeviceIsRunning, &count, &running);
   assert(err == kAudioHardwareNoError);
   if (!running) {
      fprintf(stderr, "Audio output device not running.\n");
      return -1;
   }
#endif

   /* Get format information for output device, and check that it will
      work for our purposes.
   */
   count = sizeof(devFormat);
   err = AudioDeviceGetProperty(outdevice, 0, ISOUTPUT,
                     kAudioDevicePropertyStreamFormat, &count, &devFormat);
   if (err != kAudioHardwareNoError) {
      fprintf(stderr, "Can't get audio output device format (error %ld)\n",
                                                                         err);
      return -1;
   }
   assert(devFormat.mFormatID == kAudioFormatLinearPCM);
   assert(devFormat.mFormatFlags & kLinearPCMFormatFlagIsFloat);

   if (devFormat.mSampleRate != srate) {
      fprintf(stderr, "The sampling rate used by the audio output device "
                      "(%.2f kHz) doesn't match the requested rate "
                      "(%.2f kHz).\n",
                      devFormat.mSampleRate * .001, srate * .001);
      /* And note that we can't change the dev rate (according to penrose). */
      return -1;
   }

   outdev_nchans = devFormat.mChannelsPerFrame;
   if (out_nchans > outdev_nchans) {
      fprintf(stderr, "The audio output device has %d channels, but RTcmix "
                      "wants %d channels.\n", outdev_nchans, out_nchans);
      return -1;
   }

   /* Try to set buffer size for this device in accord with RTBUFSAMPS. */
   err = AudioDeviceGetPropertyInfo(outdevice, 0, ISOUTPUT,
            kAudioDevicePropertyBufferSize, &count, &writeable);
   assert(err == kAudioHardwareNoError);
   desired_buffer_size = *nframes * outdev_nchans * sizeof(float);
   if (writeable) {
      err = AudioDeviceSetProperty(outdevice, NULL, 0, ISOUTPUT,
                        kAudioDevicePropertyBufferSize,
                        count, (void *)&desired_buffer_size);
      /* NB: even if err != kAudioHardwareNoError, we continue and use
         the default buffer size.  */
   }

   /* Get the actual buffer size.  (The device may not want to change.) */
   err = AudioDeviceGetProperty(outdevice, 0, ISOUTPUT,
                     kAudioDevicePropertyBufferSize, &count, &out_buffer_size);
   if (err != kAudioHardwareNoError) {
      fprintf(stderr, "Can't get audio output device buffer size (error %ld)\n",
                                                                         err);
      return -1;
   }
   if (out_buffer_size != desired_buffer_size) {
		int desired_frames = desired_buffer_size / outdev_nchans / sizeof(float);
      *nframes = out_buffer_size / outdev_nchans / sizeof(float);
      printf("\nWARNING: Device using buffer size of %d frames (%lu bytes),\n"
             "instead of your requested buffer size of %d frames (%lu bytes)."
             "\n\n",
             *nframes, out_buffer_size, desired_frames, desired_buffer_size);
   }

//FIXME: other sanity tests?

   /* Open input device, if necessary. */
   if (in_nchans > 0) {

      /* Get default audio input device. */
      count = sizeof(indevice);
      err = AudioHardwareGetProperty(kAudioHardwarePropertyDefaultInputDevice,
                  &count, (void *) &indevice);
      if (err != kAudioHardwareNoError || indevice == kAudioDeviceUnknown) {
         fprintf(stderr, "Can't find default audio input device (error %ld).\n",
                                                                         err);
         return -1;
      }

      /* Get format information for input device, and check it. */
      count = sizeof(devFormat);
      err = AudioDeviceGetProperty(indevice, 0, ISINPUT,
                        kAudioDevicePropertyStreamFormat, &count, &devFormat);
      if (err != kAudioHardwareNoError) {
         fprintf(stderr, "Can't get audio input device format (error %ld)\n",
                                                                         err);
         return -1;
      }
      assert(devFormat.mFormatID == kAudioFormatLinearPCM);
      assert(devFormat.mFormatFlags & kLinearPCMFormatFlagIsFloat);

      if (devFormat.mSampleRate != srate) {
         fprintf(stderr, "The sampling rate used by the audio input device "
                         "(%.2f kHz) doesn't match the requested rate "
                         "(%.2f kHz).\n",
                         devFormat.mSampleRate * .001, srate * .001);
         return -1;
      }

      indev_nchans = devFormat.mChannelsPerFrame;
      if (in_nchans > indev_nchans) {
         fprintf(stderr, "The audio input device has %d channels, but RTcmix "
                         "wants %d channels.\n", indev_nchans, in_nchans);
         return -1;
      }

      /* Try to set buffer size for this device to match *inframes. */
      err = AudioDeviceGetPropertyInfo(indevice, 0, ISINPUT,
               kAudioDevicePropertyBufferSize, &count, &writeable);
      assert(err == kAudioHardwareNoError);
      desired_buffer_size = *nframes * indev_nchans * sizeof(float);
      if (writeable) {
         err = AudioDeviceSetProperty(indevice, NULL, 0, ISINPUT,
                           kAudioDevicePropertyBufferSize,
                           count, (void *)&desired_buffer_size);
         /* ??NB: even if err != kAudioHardwareNoError, we continue and use
            the default buffer size.  */
      }

      /* Get the actual buffer size.  (The device may not want to change.) */
      err = AudioDeviceGetProperty(indevice, 0, ISINPUT,
                     kAudioDevicePropertyBufferSize, &count, &in_buffer_size);
      if (err != kAudioHardwareNoError) {
         fprintf(stderr,
               "Can't get audio input device buffer size (error %ld)\n", err);
         return -1;
      }
      if (in_buffer_size != out_buffer_size) {
         fprintf(stderr, "Input device cannot be set to use the same number "
                         "of audio frames as output device.");
         return -1;
      }

      /* For unknown reasons, I get gapping with more than one fragment when
         audio input is on.  (Could have to do with synchronization of the
         various local cur_fragment variables in this file.)   -JGG
      */
      nfrags = 1;
   }
   else
      indev_nchans = 0;

   dev_info.nfrags = nfrags;
   dev_info.nframes = *nframes;
   dev_info.in_nchans = in_nchans;
   dev_info.out_nchans = out_nchans;
   dev_info.indev_nchans = indev_nchans;
   dev_info.outdev_nchans = outdev_nchans;

   /* Adjust scheduling policy for our thread.  Using "time constraint"
      scheduling prevents GUI manipulations from causing dropouts. */
   host_clock_rate = AudioGetHostClockFrequency();
   period = (uint32_t)( ((out_buffer_size / outdev_nchans)
                                 / devFormat.mSampleRate) * host_clock_rate );
   thread_policy.period = period;
   thread_policy.computation = AudioGetHostClockMinimumTimeDelta() * 2000;
   thread_policy.constraint = thread_policy.computation * 2;
   thread_policy.preemptible = 1;
   thread_policy_set(pthread_mach_thread_np(pthread_self()),
                     THREAD_TIME_CONSTRAINT_POLICY,
                     (thread_policy_t) &thread_policy,
                     THREAD_TIME_CONSTRAINT_POLICY_COUNT);

   /* Register our callback function with the HAL. */
   err = AudioDeviceAddIOProc(outdevice, audioIOProc, (void *)&dev_info);
   assert(err == kAudioHardwareNoError);

   /* Start audio input and output devices. */
   err = AudioDeviceStart(outdevice, audioIOProc);
   assert(err == kAudioHardwareNoError);
   if (in_nchans > 0) {
      err = AudioDeviceStart(indevice, audioIOProc);
      assert(err == kAudioHardwareNoError);
   }

   /* Allocate fragmented intermediate buffers. */
   nsamps = *nframes * outdev_nchans * nfrags;
   intermediate_out_buffer = (float *)calloc(nsamps, sizeof(float));
   assert(intermediate_out_buffer != NULL);
   if (in_nchans > 0) {
      nsamps = *nframes * indev_nchans * nfrags;
      intermediate_in_buffer = (float *)calloc(nsamps, sizeof(float));
      assert(intermediate_in_buffer != NULL);
   }

   if (verbose)
      printf("Audio output buffer set to %d frames (%ld bytes).\n",
                                                *nframes, out_buffer_size);

   if (iport)
      *iport = indevice;
   if (oport)
      *oport = outdevice;

   return 0;
}

void
macosx_audio_stop(AudioDeviceID oport)
{
	OSStatus err;

	err = AudioDeviceStop(oport, audioIOProc);

	err = AudioDeviceRemoveIOProc(oport, audioIOProc);

}

#endif /* MACOSX */

